--
-- AnimationMapTriggers
--
-- @author:    	Xentro (Marcus@Xentro.se)
-- @website:	www.Xentro.se
-- @history		v1.0   - 2014-11-08 - Initial implementation
-- 				v1.0.1 - 2015-01-05 - Fixed small time flaw, show if trigger is disabled
-- 				v1.0.2 - 2015-01-30 - More error handling
-- 

function onCreate(self, id)
	local instance = AnimationMapTrigger:new(g_server ~= nil, g_client ~= nil);
	if instance:load(id) then
		g_currentMission:addOnCreateLoadedObject(instance);
		instance:register(true);
	else
		instance:delete();
	end;
end;

AnimationMapTrigger = {};

local AnimationMapTrigger_mt = Class(AnimationMapTrigger, Object);
InitObjectClass(AnimationMapTrigger, "AnimationMapTrigger");

function AnimationMapTrigger:new(isServer, isClient)
    local self = Object:new(isServer, isClient, AnimationMapTrigger_mt);
    self.className = "AnimationMapTrigger";
	
	self.triggerId = 0;
	self.isEnabled = true;
	
	self.canActivate = true;
	
	self.animationTime = 0;
	self.lastAnimationTime = -1;
	self.isRunning = false;
	
	self.forceStop = false;
	
	self.playerInRange = false;
	self.vehiclesInRange = {};
	self.numVehicles = 0;
	
	self.l10nText = {};
	
	self.animationDirtyFlag = self:getNextDirtyFlag();
	
    return self;
end;

function AnimationMapTrigger:load(id)
	if g_currentMission.AnimationMapTriggers == nil then
		g_currentMission.AnimationMapTriggers = {};
	end;
	
	local youMayPass = true;
	local name = getUserAttribute(id, "name");
	
	self.triggerId = id;
	addTrigger(self.triggerId, "triggerCallback", self);
	
	self.animationState = Utils.getNoNil(getUserAttribute(id, "animationState"), false);
	self.isPressed = Utils.getNoNil(getUserAttribute(id, "isButtonPressed"), false);
	
	local inputKey = getUserAttribute(id, "inputKey");
	if inputKey ~= nil and string.len(inputKey) >= 1 then
		if InputBinding[inputKey] ~= nil then
			self.inputKey = InputBinding[inputKey];
			local pos = getUserAttribute(id, "posText");
			
			if pos ~= nil and string.len(pos) >= 1 then
				self.l10nText[1] = pos;
			end;
			
			if self.isPressed then
				local reverseInputKey = getUserAttribute(id, "reverseInputKey");
				if reverseInputKey ~= nil and string.len(reverseInputKey) >= 1 then
					if InputBinding[reverseInputKey] ~= nil then
						self.reverseInputKey = InputBinding[reverseInputKey];
					else
						print("AnimationMapTrigger - Warning: reverseInputKey is not an valid inputBinding!");
					end;
				else
					print("AnimationMapTrigger - Error: reverseInputKey is nil!");
				end;
			else
				local neg = getUserAttribute(id, "negText");
				
				if neg ~= nil and string.len(neg) >= 1 then
					self.l10nText[2] = neg;
				end;
			end;
		else
			print("AnimationMapTrigger - Warning: InputKey is not an valid inputBinding!");
		end;
	end;
	
	if getNumOfChildren(id) >= 1 then
		local animationNode = 0; -- tonumber(getUserAttribute(id, "animationNode"));
		if animationNode ~= nil and string.len(animationNode) >= 1 then
			local rootNode = getChildAt(id, animationNode);
			self.animCharSet = 0;
			
			if rootNode ~= nil then
				self.AnimationNode = rootNode;
				self.animCharSet = getAnimCharacterSet(self.AnimationNode);
				
				if self.animCharSet ~= 0 then                    
					local animationClip = getUserAttribute(id, "animationClip");
					
					if animationClip ~= nil and string.len(animationClip) >= 1 then
						self.clip = getAnimClipIndex(self.animCharSet, animationClip);
						
						if self.clip ~= nil then
							assignAnimTrackClip(self.animCharSet, 0, self.clip);
							
							if isAnimTrackClipAssigned(self.animCharSet, 0) then
								self.loop = Utils.getNoNil(getUserAttribute(id, "loopAnimation"), false);
								
								if self.loop and self.isPressed then
									self.isPressed = false;
									print("AnimationMapTrigger - Notice: loopAnimation and isButtonPressed dont mix well! isButtonPressed have been changed to false for trigger (" .. id .. " - " .. getName(id) .. ")");
								end;
								
								self.accelerationTime = Utils.getNoNil(getUserAttribute(id, "accelerationTime"), 0) / 1000;
								self.deAccelerationTime = Utils.getNoNil(getUserAttribute(id, "deAccelerationTime"), 0) / 1000;
								self.currentAcc = 0;
								
								setAnimTrackLoopState(self.animCharSet, 0, self.loop);
								self.speedScale = Utils.getNoNil(getUserAttribute(id, "speedScale"), 1);
								setAnimTrackSpeedScale(self.animCharSet, self.clip, self.speedScale);
								self.animDuration = getAnimClipDuration(self.animCharSet, self.clip);
								
								self.hudOverlayName = getUserAttribute(id, "hudOverlayName");
							else
								print("AnimationMapTrigger - Error: clipName (" .. tostring(clip) .. ") couldn't be assigned for trigger (" .. id .. " - " .. getName(id) .. ")");
								youMayPass = false;
							end;
						else
							print("AnimationMapTrigger - Error: clipName (" .. tostring(clip) .. ") couldn't be assigned for trigger (" .. id .. " - " .. getName(id) .. ")");
							youMayPass = false;
						end;
					else
						print("AnimationMapTrigger - Error: animationClip is empty for trigger (" .. id .. " - " .. getName(id) .. ")");
						youMayPass = false;
					end;
				else
					print("AnimationMapTrigger - Error: Can't load animation clip, possible reason can be wrong animationClip or wrong animationNode for trigger (" .. id .. " - " .. getName(id) .. ")");
					youMayPass = false;
				end;
			else
				print("AnimationMapTrigger - Error: animationNode is nil for trigger (" .. id .. " - " .. getName(id) .. ").");
				youMayPass = false;
			end;
		else
			youMayPass = false;
		end;
	else
		print("AnimationMapTrigger - Error: trigger (" .. id .. " - " .. getName(id) .. ") haven't been properly setup! there are no children of the trigger.");
		youMayPass = false;
	end;
	
	self.showDisableText = Utils.getNoNil(getUserAttribute(id, "showDisableText"), true);
	local disableText = getUserAttribute(id, "disableText");
	if disableText ~= nil and string.len(disableText) >= 1 then
		self.l10nText[3] = disableText;
	else
		self.l10nText[3] = "TRIGGER_DISABLED";
	end;
	
	self.openTime = Utils.getNoNil(getUserAttribute(id, "openTime"), 0);
	self.closeTime = Utils.getNoNil(getUserAttribute(id, "closeTime"), 0);
	self.disableOnClose = Utils.getNoNil(getUserAttribute(id, "disableOnClose"), true);
	
	if youMayPass then
		g_currentMission:addOnCreateLoadedObjectToSave(self);
		
		if name ~= nil then
			g_currentMission.AnimationMapTriggers[name] = self;
		else
			table.insert(g_currentMission.AnimationMapTriggers, self);
		end;
	end;
	
	return youMayPass;
end;

function AnimationMapTrigger:finishLoad()
	local soundFile = getUserAttribute(self.triggerId, "soundFile");
	
	if g_currentMission.AnimationMapConfig ~= nil then
		if soundFile ~= nil and string.len(soundFile) >= 1 then
			self.animSoundPath = Utils.getFilename(soundFile, g_currentMission.AnimationMapConfig["mapDir"]);
			
			if fileExists(self.animSoundPath) then
				self.animSoundVolume = Utils.getNoNil(getUserAttribute(self.triggerId, "animSoundVolume"), 1);
				self.animSoundRadius = Utils.getNoNil(getUserAttribute(self.triggerId, "animSoundRadius"), 15);
				self.animSoundInnerRadius = Utils.getNoNil(getUserAttribute(self.triggerId, "animSoundInnerRadius"), 5);
				local loopInt = 0;
				local loop = getUserAttribute(self.triggerId, "playSoundOnlyOnce");
				if loop ~= nil and loop then
					loopInt = 1;
				end;
				
				self.clientAnimationSound = createAudioSource(soundFile, self.animSoundPath, self.animSoundRadius, self.animSoundInnerRadius, self.animSoundVolume, loopInt);
				
				link(self.AnimationNode, self.clientAnimationSound);
				setVisibility(self.clientAnimationSound, false);
				
				self.playAnimationSoundEnabled = false;
			else
				print("AnimationMapTrigger - Warning: Can't load soundFile " .. tostring(self.animSoundPath));
			end;
		end;
		
		
		if g_currentMission.AnimationMapConfig["debug"] ~= nil and g_currentMission.AnimationMapConfig["debug"] then
			local id = 0;
			
			for i, s in pairs(g_currentMission.AnimationMapTriggers) do
				if s == self then
					id = i;
					break;
				end;
			end;
			
			print("AnimationMapTrigger - Debug: Trigger " .. getName(self.triggerId) .. " with table id of " .. id .. " have been loaded.");
		end;
	end;
end;

function AnimationMapTrigger:delete()
	g_currentMission:removeOnCreateLoadedObjectToSave(self);
	
	if self.triggerId ~= 0 then
		removeTrigger(self.triggerId);
	end;
end;

function AnimationMapTrigger:readStream(streamId, connection)
	if connection:getIsServer() then
		self.animationTime = streamReadFloat32(streamId);
		local state = streamReadBool(streamId);
		
		self:setAnimation(state, true);
		
		setAnimTrackTime(self.animCharSet, self.clip, self.animationTime);
	end;
end;

function AnimationMapTrigger:writeStream(streamId, connection)
	if not connection:getIsServer() then
		streamWriteFloat32(streamId, self.animationTime);
		streamWriteBool(streamId, self.animationState);
	end;
end;

function AnimationMapTrigger:readUpdateStream(streamId, timestamp, connection)
	local hasUpdate = streamReadBool(streamId);
	
	if hasUpdate then
		self:readStream(streamId, connection);
	end;
end;

function AnimationMapTrigger:writeUpdateStream(streamId, connection, dirtyMask)
	local sendUpdate = streamWriteBool(streamId, bitAND(dirtyMask, self.animationDirtyFlag) ~= 0);
	
	if sendUpdate then
		self:writeStream(streamId, connection);
	end;
end;

function AnimationMapTrigger:loadFromAttributesAndNodes(xmlFile, tag)
	local state = getXMLBool(xmlFile, tag .. "#animationState");
	local time = getXMLFloat(xmlFile, tag .. "#animationTime");
	local isRunning = getXMLBool(xmlFile, tag .. "#isRunning");
	
	if state ~= nil then
		self.animationState = state;
	end;
	if time ~= nil then
		self.animationTime = time;
		
		enableAnimTrack(self.animCharSet, self.clip);
		setAnimTrackTime(self.animCharSet, self.clip, self.animationTime, true);
		disableAnimTrack(self.animCharSet, self.clip);
	end;
	
	if isRunning ~= nil and isRunning then
		self:setAnimation(self.animationState, true);
	end;
	
	return true;
end;

function AnimationMapTrigger:getSaveAttributesAndNodes()
	local attributes = 'animationState="' .. tostring(self.animationState) .. '" animationTime="' .. self.animationTime .. '" isRunning="' .. tostring(self.isRunning) ..'"';
	local nodes = "";
	
	return attributes, nodes;
end;

function AnimationMapTrigger:update(dt)
	if self.inputKey ~= nil then
		local activeTrigger, isInTrigger = self:getCanInteract();
		
		if activeTrigger then
			if self.isPressed then
				if g_gui.currentGui == nil and not g_currentMission.isPlayerFrozen then
					if InputBinding.isPressed(self.inputKey) then
						self:setAnimation(true);
					elseif self.reverseInputKey ~= nil and InputBinding.isPressed(self.reverseInputKey) then
						self:setAnimation(false);
					else
						self.forceStop = true;
					end;
				else
					self.forceStop = true;
				end;
				
				if not self.isRunning then
					if self.l10nText[1] ~= nil then
						g_currentMission:addHelpButtonText(g_i18n:getText(self.l10nText[1]), self.inputKey);
					end;
				end;
			else
				if g_gui.currentGui == nil and not g_currentMission.isPlayerFrozen then
					if InputBinding.hasEvent(self.inputKey) then
						self:setAnimation(not self.animationState);
					end;
				end;
				
				if self.animationState then
					if self.l10nText[2] ~= nil then
						g_currentMission:addHelpButtonText(g_i18n:getText(self.l10nText[2]), self.inputKey);
					end;
				else
					if self.l10nText[1] ~= nil then
						g_currentMission:addHelpButtonText(g_i18n:getText(self.l10nText[1]), self.inputKey);
					end;
				end;
			end;
		else
			if isInTrigger and self.showDisableText then
				g_currentMission:addExtraPrintText(g_i18n:getText(self.l10nText[3]));
			end;
		end;
	end;
	
	if self.clientAnimationSound ~= nil then
		if self.isRunning then
			if g_client ~= nil then
				if not self.playAnimationSoundEnabled then
					setVisibility(self.clientAnimationSound, true);
					self.playAnimationSoundEnabled = true;
				end;
			end;
		else
			if self.playAnimationSoundEnabled then
				setVisibility(self.clientAnimationSound, false);
				self.playAnimationSoundEnabled = false;
			end;
		end;
	end;
end;

function AnimationMapTrigger:updateTick(dt)
	if self.inputKey == nil and self.canActivate and (self.openTime == 0 or self.closeTime == 0) then
		if self.numVehicles > 0 then
			self:setAnimation(true, true);
		else
			self:setAnimation(false, true);
		end;
	end;
	
	if self.openTime ~= 0 and self.closeTime ~= 0 then
		if g_currentMission.environment.currentHour >= self.openTime and g_currentMission.environment.currentHour < self.closeTime then
			if self.objectIsClosed ~= nil then
				if not self.animationState then
					self:setAnimation(true, true);
				end;
				
				self.objectIsClosed = nil;
			end;
			
			if self.disableOnClose then
				self.canActivate = true;
			end;
		else
			if self.objectIsClosed == nil then
				if self.animationState then
					self:setAnimation(false, true);
				end;
				
				self.objectIsClosed = true;
			end;
			
			if self.disableOnClose then
				self.canActivate = false;
			end;
		end;
	end;
	
	if self.isRunning then
		self.animationTime = getAnimTrackTime(self.animCharSet, self.clip);
		local stopClip = self.forceStop;
		
		if self.loop then
			if self.animationState then
				if self.accelerationTime ~= 0 then
					if self.currentAcc < 1 then
						self.currentAcc = self.currentAcc + dt * self.accelerationTime;
						setAnimTrackSpeedScale(self.animCharSet, self.clip, self.currentAcc);
					end;
				end;
			else
				if self.deAccelerationTime ~= 0 then
					if self.currentAcc > 0 then
						self.currentAcc = self.currentAcc - dt * self.deAccelerationTime;
						setAnimTrackSpeedScale(self.animCharSet, self.clip, self.currentAcc);
					else
						stopClip = true;
					end;
				else
					stopClip = true;
				end;
			end;
		else
			if self.animationTime > self.animDuration then
				setAnimTrackTime(self.animCharSet, self.clip, self.animDuration);
			elseif self.animationTime < 0 then
				setAnimTrackTime(self.animCharSet, self.clip, 0);
			end;
			
			if self.animationState then
				if self.animationTime > self.animDuration then
					stopClip = true;
				end;
			else
				if self.animationTime < 0 then
					stopClip = true;
				end;
			end;
		end;
		
		if stopClip then
			disableAnimTrack(self.animCharSet, self.clip);
			self.isRunning = false;
		end;
	end;
	
	if g_server ~= nil then
		if self.animationTime ~= self.lastAnimationTime then
			self:raiseDirtyFlags(self.animationDirtyFlag);
			
			self.lastAnimationTime = self.animationTime;
		end;
	end;
end;

function AnimationMapTrigger:draw()
	if self.inputKey ~= nil then
		local activeTrigger, isInTrigger = self:getCanInteract();
		
		if activeTrigger then
			if self.hudOverlayName ~= nil and g_currentMission.AnimationMapHuds[self.hudOverlayName] ~= nil then
				g_currentMission.AnimationMapHuds[self.hudOverlayName]:render();
			end;
		end;
	end;
end;

function AnimationMapTrigger:getCanInteract()
	local canActivate = false;
		
	if (g_currentMission.controlPlayer and self.playerInRange) then
		canActivate = true;
	end;
	
	if not canActivate then
		if not g_currentMission.controlPlayer then
			for vehicle in pairs(self.vehiclesInRange) do
				if vehicle:getIsActiveForInput(false) then
					canActivate = true;
					break;
				end;
			end;
		end;
	end;
	
	if not self.canActivate then
		return false, canActivate;
	end;
	
	return canActivate;
end;

function AnimationMapTrigger:setAnimation(state, noEventSend)
	if self.animationState ~= state or not self.isRunning then
		setAnimationClipEvent.sendEvent(self, state, noEventSend);
		
		self.animationState = state;
		self.forceStop = false;
		
		if not self.loop then
			local speedScale = self.speedScale;
			if not state then
				speedScale = -speedScale;
			end;
			
			setAnimTrackSpeedScale(self.animCharSet, self.clip, speedScale);
			enableAnimTrack(self.animCharSet, self.clip);
			self.isRunning = true;
		else
			if state then
				if self.accelerationTime == 0 then
					self.currentAcc = 1;
					setAnimTrackSpeedScale(self.animCharSet, self.clip, self.currentAcc);
				end;
			else
				if self.deAccelerationTime == 0 then
					self.currentAcc = 0;
					disableAnimTrack(self.animCharSet, self.clip);
					self.isRunning = false;
				end;
			end;
			
			enableAnimTrack(self.animCharSet, self.clip);
			self.isRunning = true;
		end;
	end;
end;

function AnimationMapTrigger:triggerCallback(triggerId, otherId, onEnter, onLeave, onStay, otherShapeId)
	if self.isEnabled then
		if (onEnter or onLeave) then
			if g_currentMission.player ~= nil and otherId == g_currentMission.player.rootNode then
				if onEnter then
					self.playerInRange = true;
				else
					self.playerInRange = false;
				end;
			else
				local vehicle = g_currentMission.nodeToVehicle[otherId];
				if vehicle ~= nil then
					if onEnter then
						if self.vehiclesInRange[vehicle] == nil then
							self.vehiclesInRange[vehicle] = true;
							self.numVehicles = self.numVehicles + 1; 
						end;
					else
						if self.vehiclesInRange[vehicle] then
							self.vehiclesInRange[vehicle] = nil;
							self.numVehicles = math.max(self.numVehicles - 1, 0); 
						end;
					end;
				end;
			end;
		end;
	end;
end;


-- Event
setAnimationClipEvent = {};
setAnimationClipEvent_mt = Class(setAnimationClipEvent, Event);

InitEventClass(setAnimationClipEvent, "setAnimationClipEvent");

function setAnimationClipEvent:emptyNew()
	local self = Event:new(setAnimationClipEvent_mt);
	
    return self;
end;

function setAnimationClipEvent:new(object, isActivated)
	local self = setAnimationClipEvent:emptyNew();
	self.object = object;
	self.isActivated = isActivated;
	
	return self;
end;

function setAnimationClipEvent:readStream(streamId, connection)
	local id = streamReadInt32(streamId);
	self.object = networkGetObject(id);
	self.isActivated = streamReadBool(streamId);
	
    self:run(connection);
end;

function setAnimationClipEvent:writeStream(streamId, connection)
	streamWriteInt32(streamId, networkGetObjectId(self.object));
	streamWriteBool(streamId, self.isActivated);
end;

function setAnimationClipEvent:run(connection)
	self.object:setAnimation(self.isActivated, true);
	
	if not connection:getIsServer() then
		g_server:broadcastEvent(setAnimationClipEvent:new(self.object, self.isActivated), nil, connection, self.object);
	end;
end;

function setAnimationClipEvent.sendEvent(object, isActivated, noEventSend)
	if noEventSend == nil or noEventSend == false then
		if g_server ~= nil then
			g_server:broadcastEvent(setAnimationClipEvent:new(object, isActivated), nil, nil, object);
		else
			g_client:getServerConnection():sendEvent(setAnimationClipEvent:new(object, isActivated));
		end;
	end;
end;