--
-- Mulcher
-- Class for all mulchers
--
-- @author  Stefan Geiger
-- @date  11/05/08
--
-- Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
     
Mulcher = {};
    
function Mulcher.prerequisitesPresent(specializations)
    return true;
end;
    
function Mulcher:load(xmlFile)
    
	self.setIsTurnedOn = SpecializationUtil.callSpecializationsFunction("setIsTurnedOn");
	
	self.groundReferenceThreshold = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.groundReferenceNode#threshold"), 0.2);
	self.groundReferenceNode = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.groundReferenceNode#index"));
	if self.groundReferenceNode == nil then
		self.groundReferenceNode = self.components[1].node;
	end;
    
	if self.isClient then
		local mulcherStartSound = getXMLString(xmlFile, "vehicle.mulcherStartSound#file");
		if mulcherStartSound ~= nil and mulcherStartSound ~= "" then
			mulcherStartSound = Utils.getFilename(mulcherStartSound, self.baseDirectory);
			self.mulcherStartSound = createSample("mulcherStartSound");
			loadSample(self.mulcherStartSound, mulcherStartSound, false);
			self.mulcherStartSoundPitchOffset = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.mulcherStartSound#pitchOffset"), 1);
			self.mulcherStartSoundVolume = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.mulcherStartSound#volume"), 1);
			setSamplePitch(self.mulcherStartSound, self.mulcherStartSoundPitchOffset);
		end;
		self.mulcherSoundEnabled = false;

		local mulcherSound = getXMLString(xmlFile, "vehicle.mulcherSound#file");
		if mulcherSound ~= nil and mulcherSound ~= "" then
			mulcherSound = Utils.getFilename(mulcherSound, self.baseDirectory);
			self.mulcherSound = createSample("mulcherSound");
			loadSample(self.mulcherSound, mulcherSound, false);
			self.mulcherSoundPitchOffset = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.mulcherSound#pitchOffset"), 1);
			self.mulcherSoundVolume = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.mulcherSound#volume"), 1);
			setSamplePitch(self.mulcherSound, self.mulcherSoundPitchOffset);

			self.mulcherSound3DVolume = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.mulcherSound#volume3D"), self.mulcherSoundVolume);
			self.mulcherSound3DInnerRadius = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.mulcherSound#innerRadius"), 10);
			self.mulcherSound3DRadius = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.mulcherSound#radius"), 50);

			self.mulcherSound3D = createAudioSource("mulcherSound3D", mulcherSound, self.mulcherSound3DRadius, self.mulcherSound3DInnerRadius, self.mulcherSound3DVolume, 0);
			setSamplePitch(getAudioSourceSample(self.mulcherSound3D), self.mulcherSoundPitchOffset);
			link(self.components[1].node, self.mulcherSound3D);
			setVisibility(self.mulcherSound3D, false);
		end;

		local mulcherStopSound = getXMLString(xmlFile, "vehicle.mulcherStopSound#file");
		if mulcherStopSound ~= nil and mulcherStopSound ~= "" then
			mulcherStopSound = Utils.getFilename(mulcherStopSound, self.baseDirectory);
			self.mulcherStopSound = createSample("mulcherStopSound");
			loadSample(self.mulcherStopSound, mulcherStopSound, false);
			self.mulcherStopSoundPitchOffset = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.mulcherStopSound#pitchOffset"), 1);
			self.mulcherStopSoundVolume = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.mulcherStopSound#volume"), 1);
			setSamplePitch(self.mulcherStopSound, self.mulcherStopSoundPitchOffset);
		end;
    end;
  
	self.turnedOnRotationNodes = {};
	local i = 0;
	while true do
		local key = string.format("vehicle.turnedOnRotationNodes.turnedOnRotationNode(%d)", i);
		if not hasXMLProperty(xmlFile, key) then
			break;
		end;
		local node = Utils.indexToObject(self.components, getXMLString(xmlFile, key.."#index"));
		local x,y,z = Utils.getVectorFromString(getXMLString(xmlFile, key.."#rotSpeed"))
		local curSpeed = {0,0,0};
		local minSpeed = {0,0,0};
		local maxSpeed = {x,y,z};
		local idle = Utils.getNoNil(getXMLInt(xmlFile, key.."#idle"), 1);
		if node ~= nil then
			table.insert(self.turnedOnRotationNodes, {node=node, curSpeed=curSpeed, minSpeed = minSpeed, maxSpeed = maxSpeed, idle = idle});
		end;
		i = i + 1;
	end;
	
	self.mulcherTurnOnAnimation = getXMLString(xmlFile, "vehicle.mulcherTurnOnAnimation#name");
	self.mulcherTurnOnAnimationSpeed = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.mulcherTurnOnAnimation#turnOnSpeed"), 1);
	self.mulcherTurnOffAnimationSpeed = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.mulcherTurnOnAnimation#turnOffSpeed"), -self.mulcherTurnOnAnimationSpeed);

	-- ps - ParticleSystem
	self.showParticle = false;
	self.particleSystems = {};
	local i = 0;
        while true do
            local key = string.format("vehicle.particleSystems.system(%d)", i);
            local t = getXMLString(xmlFile, key .. "#type");
            if t == nil then
                break;
            end;
            local desc = FruitUtil.fruitTypes[t];
            if desc ~= nil then
                local currentPS = {};
				local index = Utils.indexToObject(self.components, getXMLString(xmlFile, key .. "#index"));
                local particleNode = Utils.loadParticleSystem(xmlFile, currentPS, key, self.components, false, "particleSystems/dust.i3d", self.baseDirectory, index);
                self.particleSystems[desc.index] = currentPS;
				self.particleSystems.index = index;
            end;
        i = i + 1;
    end;
	
	self.dustParticle = {};
	self.showDust = false;
	local index = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.particleSystems.dust#index"));
	Utils.loadParticleSystem(xmlFile, self.dustParticle, "vehicle.particleSystems.dust", self.components, false, nil, self.baseDirectory, index);
	
	
	self.backupFruitType = {};
	self.fruitType = {};
	self.fruitModes = {};
	local i = 0;
	while true do
		local key = string.format("vehicle.fruitModes.fruit(%d)", i);
		local name = getXMLString(xmlFile, key.."#name");
		if name == nil then
			break;
		end;
		local desc = FruitUtil.fruitTypes[name];
		if desc ~= nil then
			local mode = Utils.getNoNil(getXMLInt(xmlFile, key.."#mode"), 3);
			if mode > 3 then
				print("Error : invalid fruitMode for "..tostring(name).." in "..tostring(self.name));
				mode = 3;
			end;
			self.fruitModes[desc.index] = mode;
		end;
		i = i+1;
	end;
 
    self.isTurnedOn = false;
    self.wasToFast = false;
 
    self.mulcherGroundFlag = self:getNextDirtyFlag();
 end;
 
function Mulcher:delete()
    if self.mulcherStartSound ~= nil then
        delete(self.mulcherStartSound);
    end;
	
    if self.mulcherSound ~= nil then
        delete(self.mulcherSound);
        self.mulcherSoundEnabled = false;
    end;
	
    if self.mulcherStopSound ~= nil then
        delete(self.mulcherStopSound);
    end; 
	
	for fruitType,_ in pairs(g_currentMission.fruits) do
		Utils.deleteParticleSystem(self.particleSystems[fruitType]);
	end;
	Utils.deleteParticleSystem(self.dustParticle);
end;
 
function Mulcher:readStream(streamId, connection)
    local turnedOn = streamReadBool(streamId);
    self:setIsTurnedOn(turnedOn, true);
 end;
 
function Mulcher:writeStream(streamId, connection)
    streamWriteBool(streamId, self.isTurnedOn);
end;
 
function Mulcher:readUpdateStream(streamId, timestamp, connection)
end;
 
function Mulcher:writeUpdateStream(streamId, connection, dirtyMask)
end;
 
function Mulcher:mouseEvent(posX, posY, isDown, isUp, button)
end;
 
function Mulcher:keyEvent(unicode, sym, modifier, isDown)
end;
 
function Mulcher:update(dt) 
    if self:getIsActive() then
        if self:getIsActiveForInput() then
            if InputBinding.hasEvent(InputBinding.IMPLEMENT_EXTRA) then
                self:setIsTurnedOn(not self.isTurnedOn);
            end;
        end;
 
			for _, node in pairs(self.turnedOnRotationNodes) do
 				local values = Utils.getMovedLimitedValues(node.curSpeed, node.maxSpeed, node.minSpeed, 3, node.idle*2500, dt, not self.isTurnedOn);
				node.curSpeed = values;
				rotate(node.node, unpack(node.curSpeed));
                if self.setMovingToolDirty ~= nil then
                    self:setMovingToolDirty(node.node);
                end
 			end
 	
    end
end;
 
function Mulcher:updateTick(dt)
    self.wasToFast = false;
	self.showParticle = false;
		
    if self:getIsActive() then
        if self.isTurnedOn then
            if self.movingDirection < 0 then 
				setRotation(self.particleSystems.index, 0, 180, 0);
			else
				setRotation(self.particleSystems.index, 0, 0, 0);
			end;
			
			local toFast = self:doCheckSpeedLimit() and self.lastSpeed*3600 > 31;
            if not toFast then
                if self.isServer then
                    local x,y,z = getWorldTranslation(self.groundReferenceNode);
                    local terrainHeight = getTerrainHeightAtWorldPos(g_currentMission.terrainRootNode, x, 0, z);
                    if terrainHeight+self.groundReferenceThreshold >= y then
                        local foldAnimTime = self.foldAnimTime;

                        local numAreasSend = 0;
                        local cuttingAreasSend = {};
                        for _,area in pairs(self.cuttingAreas) do
                            if self:getIsAreaActive(area) then
                                numAreasSend = numAreasSend + 1;
				
								local x,y,z = getWorldTranslation(area.start);
								local x1,y1,z1 = getWorldTranslation(area.width);
								local x2,y2,z2 = getWorldTranslation(area.height);
							
								table.insert(cuttingAreasSend, {x,z,x1,z1,x2,z2});
								
								for fruitType,_ in pairs(g_currentMission.fruits) do
									local checkFruit = Utils.getFruitArea(fruitType, x, z, x1, z1, x2, z2);
									local checkWindrow = Utils.getFruitWindrowArea(fruitType, x, z, x1, z1, x2, z2);
									local checkCutLong = Utils.getFruitCutLongArea(fruitType, x, z, x1, z1, x2, z2);
									
									if checkFruit > 0 or checkWindrow > 0 or checkCutLong > 0 then
										mode = self.fruitModes[fruitType];
										if checkWindrow > 0 then
											mode = MulcherEvent.DESTROYWINDROW;											
										end;
										self.fruitType = fruitType;
										self.showParticle = true;
										break;
									end;
								end;						
                            end
                        end                 
                        if table.getn(cuttingAreasSend) > 0 then                          
                            MulcherEvent.runLocally(cuttingAreasSend, false, mode, self.fruitType);
							--g_server:broadcastEvent(MulcherEvent:new(areasSend, false, self.fruitType));
                        end;						
						self.showDust = true;
					else
						self.showDust = false;			
                    end;
                end;
            end;

            if self.isClient then
                if self:getIsActiveForSound() then
                    if self.mulcherSound ~= nil then
                        if not self.mulcherSoundEnabled then
                            if self.mulcherStartSound == nil or not isSamplePlaying(self.mulcherStartSound) or getSamplePlayOffset(self.mulcherStartSound) >= getSampleDuration(self.mulcherStartSound)-1.8*dt then
                                playSample(self.mulcherSound, 0, self.mulcherSoundVolume, 0);
                                self.mulcherSoundEnabled = true;
                            end
                        end
                    end;
                    if self.mulcherSound3D ~= nil then
                        setVisibility(self.mulcherSound3D, false);
                    end
                else
                    if self.mulcherSound3D ~= nil then
                        setVisibility(self.mulcherSound3D, true);
                    end
                end
            end;
            self.wasToFast = toFast;
        end;
		
		if self.fruitType ~= self.backupFruittype then
			for fruitType,_ in pairs(g_currentMission.fruits) do
				Utils.setEmittingState(self.particleSystems[fruitType], false);
			end;
			self.backupFruitType = self.fruitType;
		end;
		if self.movingDirection ~= 0 then
			Utils.setEmittingState(self.particleSystems[self.fruitType], self.showParticle);
		end;
		local rain = g_currentMission.environment.lastRainScale <= 0.1 and g_currentMission.environment.timeSinceLastRain > 30;
		Utils.setEmittingState(self.dustParticle, (rain and self.isTurnedOn and self.showDust))
    end;
end;

function Mulcher:draw()
    if self.isClient then
        if self:getIsActiveForInput(true) then
            if self.isTurnedOn then
                g_currentMission:addHelpButtonText(string.format(g_i18n:getText("turn_off_OBJECT"), self.typeDesc), InputBinding.IMPLEMENT_EXTRA);
            else
                g_currentMission:addHelpButtonText(string.format(g_i18n:getText("turn_on_OBJECT"), self.typeDesc), InputBinding.IMPLEMENT_EXTRA);
            end;           
        end
         if self.wasToFast then
            g_currentMission:addWarning(g_i18n:getText("Dont_drive_to_fast") .. "\n" .. string.format(g_i18n:getText("Cruise_control_levelN"), "2"), 0.07+0.022, 0.019+0.029);
        end;
    end;
end;

function Mulcher:onDetach()
    if self.deactivateOnDetach then
        Mulcher.onDeactivate(self);
    else
        Mulcher.onDeactivateSounds(self)
    end;
end;

function Mulcher:onLeave()
    if self.deactivateOnLeave then
        Mulcher.onDeactivate(self);
    else
        Mulcher.onDeactivateSounds(self)
    end;
end;

function Mulcher:onDeactivate()
    Mulcher.onDeactivateSounds(self)
    self:setIsTurnedOn(false, true);
	
	Utils.setEmittingState(self.dustParticle, false);
	for fruitType,_ in pairs(g_currentMission.fruits) do
		Utils.setEmittingState(self.particleSystems[fruitType], false);
	end;
end;

function Mulcher:onDeactivateSounds()
    if self.isClient then
        if self.mulcherSoundEnabled then
            stopSample(self.mulcherSound);
            self.mulcherSoundEnabled = false;
        end;
        if self.mulcherSound3D ~= nil then
            setVisibility(self.mulcherSound3D, false);
        end
    end;
end;

function Mulcher:setIsTurnedOn(isTurnedOn, noEventSend)
    if isTurnedOn ~= self.isTurnedOn then
        SetTurnedOnEvent.sendEvent(self, isTurnedOn, noEventSend)
        self.isTurnedOn = isTurnedOn;


        if self.isClient then
            if self.mulcherSoundEnabled then
                stopSample(self.mulcherSound);
                self.mulcherSoundEnabled = false;
            end
            if self.mulcherStartSound ~= nil then
                stopSample(self.mulcherStartSound);
            end
            if self.mulcherStopSound ~= nil then
                stopSample(self.mulcherStopSound);
             end
 
            if self.mulcherSound3D ~= nil then
                setVisibility(self.mulcherSound3D, false);
            end

            if self:getIsActiveForSound() then
                if self.isTurnedOn then
                    if self.mulcherStartSound ~= nil then
                        playSample(self.mulcherStartSound, 1, self.mulcherStartSoundVolume, 0);
                    end
                else
                    if self.mulcherStopSound ~= nil then
                        playSample(self.mulcherStopSound, 1, self.mulcherStopSoundVolume, 0);
                    end
                end
            end
        end
 
        if self.mulcherTurnOnAnimation ~= nil and self.playAnimation ~= nil then
            local speed = self.mulcherTurnOnAnimationSpeed;
            if not self.isTurnedOn then
                speed = self.mulcherTurnOffAnimationSpeed;
            end
            self:playAnimation(self.mulcherTurnOnAnimation, speed, nil, true);
        end;
    end
end;