--
-- WheelShader
--
-- @author:        Xentro (Marcus@Xentro.se)
-- @website:    www.Xentro.se
-- @history:    v0.8 - 2016-01-03 - Test version
-- 

WheelShader = {};

WheelShader.UPDATE_TIME = 3; -- How oftan to update wheel, is 3 slow enough for MP?

function WheelShader.prerequisitesPresent(specializations)
    return true;
end;

function WheelShader:load(savegame)
    self.updateshaderWheelPosition = WheelShader.updateshaderWheelPosition;
    self.loadDataFromXML = WheelShader.loadDataFromXML;
    self.loadWheelDataFromXML = WheelShader.loadWheelDataFromXML;
    
    if self.weAreHardPointObject then
        self.ourWheelParent = self.vehicle;
        self.firstTimeRun = false;
    else
        self.ourWheelParent = self;
    end;
    
    self.curveSettings = {};
    -- default curve
    self.curveSettings["spring"] = AnimCurve:new(linearInterpolator1);
    self.curveSettings["spring"]:addKeyframe({v = 1.0, time = 0});
    self.curveSettings["spring"]:addKeyframe({v = 0.7, time = 0.4});
    self.curveSettings["spring"]:addKeyframe({v = 0.7, time = 0.7});
    self.curveSettings["spring"]:addKeyframe({v = 1.0, time = 1});
        
    self.curveSettings["frictionScale"] = AnimCurve:new(linearInterpolator1);
    self.curveSettings["frictionScale"]:addKeyframe({v = 0.2, time = 0});
    self.curveSettings["frictionScale"]:addKeyframe({v = 1.4, time = 0.4});
    self.curveSettings["frictionScale"]:addKeyframe({v = 0.9, time = 1.0});

    -- load custom curves if any..
    for _, name in pairs({"spring", "frictionScale", "damper", "maxLongStiffness", "maxLatStiffness", "maxLatStiffnessLoad"}) do
        WheelShader.loadCurve(self, xmlFile, self.curveSettings, name);
    end;
    
    self:loadDataFromXML(xmlFile);
    
    self.forceShaderWheelUpdate = false;
    
    self.dynamicLoadedShaderWheels = {};
    self.shaderWheels = {};
    local i = 0;
    while true do
        local key = string.format("vehicle.shaderWheels.wheel(%d)", i);
        if not hasXMLProperty(xmlFile, key) then break;    end;
        
        local wheelID = getXMLInt(xmlFile, key .. "#id");
        
        if wheelID ~= nil then
            local wheel = self.ourWheelParent.wheels[wheelID];
            
            if wheel ~= nil then
                local node;
                
                local filename = getXMLString(xmlFile, key .. "#filename");
                
                if filename ~= nil and filename ~= "" then
                    local dynamicallyLoadedPart = {filename = filename};
                    local i3dNode = Utils.loadSharedI3DFile(filename, self.baseDirectory, false, false, false);
                    
                    if i3dNode ~= 0 then
                        node = Utils.indexToObject(i3dNode, getXMLString(xmlFile, key .. "#node"));
                        
                        if node ~= nil then
                            local linkNode = Utils.indexToObject(self.components, Utils.getNoNil(getXMLString(xmlFile, key .. "#linkNode"), "0>"));

                            local x, y, z = Utils.getVectorFromString(getXMLString(xmlFile, key .. "#position"));
                            if x ~= nil and y ~= nil and z ~= nil then
                                setTranslation(node, x, y, z);
                            end;
                            
                            local rotX, rotY, rotZ = Utils.getVectorFromString(getXMLString(xmlFile, key .. "#rotation"));
                            if rotX ~= nil and rotY ~= nil and rotZ ~= nil then
                                rotX = Utils.degToRad(rotX);
                                rotY = Utils.degToRad(rotY);
                                rotZ = Utils.degToRad(rotZ);
                                setRotation(node, rotX, rotY, rotZ);
                            end;

                            local shaderParameterName = getXMLString(xmlFile, key .. "#shaderParameterName");
                            local x, y, z, w = Utils.getVectorFromString(getXMLString(xmlFile, key .. "#shaderParameter"));
                            if shaderParameterName ~= nil and x ~= nil and y ~= nil and z ~= nil and w ~= nil then
                                setShaderParameter(node, shaderParameterName, x, y, z, w, false);
                            end;

                            link(linkNode, node);
                            delete(i3dNode);
                            table.insert(self.dynamicLoadedShaderWheels, dynamicallyLoadedPart);
                        end;
                    end;
                else
                    node = Utils.indexToObject(self.components, getXMLString(xmlFile, key .. "#node"));
                end;
                
                if node ~= nil then
                    if getHasShaderParameter(node, "wheelParameters") then
                        local entry = {};
                        entry.node = node;
                        entry.steeringNode = Utils.indexToObject(self.components, getXMLString(xmlFile, key .. "#steeringNode"));
                        entry.driveNode = Utils.indexToObject(self.components, getXMLString(xmlFile, key .. "#driveNode"));
                        entry.fenderNode = Utils.indexToObject(self.components, getXMLString(xmlFile, key .. "#fenderNode"));
        
                        entry.wheelDiameter = wheel.radius * 2;
                        
                        entry.deformation = 0;
                        entry.suspensionCompression = 0;
                        entry.lastDeformationWithGroundContact = 0;
                        entry.lastSuspensionCompression = -1;
                        entry.lastPositionUpdate = -1;
                        
                        entry.currentRotation = 0;
                        entry.sentRotation = 0;
                        
                        entry.wheelId = wheelID;
                        entry.xmlIndex = i;
                        
                        self:loadWheelDataFromXML(xmlFile, key, entry);
                        
                        setShaderParameter(entry.node, "updateWheel", entry.deformation, 0, entry.maxDeformation, 0, false);
                        setShaderParameter(entry.node, "wheelParameters", entry.affectedRadius, entry.innerRadius, wheel.radius, entry.deformationY, false);
                        setShaderParameter(entry.node, "factorValues", entry.centerFactor, entry.extraPressureFactors[2], entry.extraPressureFactors[2], entry.extraPressureFactors[3], false);
                        
                        self:updateshaderWheelPosition(entry, wheel);
                        
                        table.insert(self.shaderWheels, entry);
                    else
                        print("  WheelShader - Error: Invalid shader wheel node, it don't have the shader connected!");
                        break;
                    end;
                end;
            else
                print("  WheelShader - Error: Invalid wheel id ( " .. wheelID .. ")");
                break;
            end;
        end;
        
        i = i + 1;
    end;
    
    self.wheelShaderDirtyFlag = self:getNextDirtyFlag();
end;

function WheelShader:delete()
    for _, dynamicallyLoadedPart in pairs(self.dynamicLoadedShaderWheels) do
        if dynamicallyLoadedPart.filename ~= nil then
            Utils.releaseSharedI3DFile(dynamicallyLoadedPart.filename, self.baseDirectory, true);
        end;
    end;
end;

function WheelShader:readStream(streamId, connection)
end;

function WheelShader:writeStream(streamId, connection)
end;

function WheelShader:readUpdateStream(streamId, timestamp, connection)
    if connection:getIsServer() then -- client
        if streamReadBool(streamId) then
            for _, v in ipairs(self.shaderWheels) do
                local rotation = streamReadFloat32(streamId);
                
                v.currentRotation = rotation;
                setShaderParameter(v.node, "rotateWheel", 0, 0, 0, v.currentRotation, false);
            end;
        end;
    end;
end;

function WheelShader:writeUpdateStream(streamId, connection, dirtyMask)
    if not connection:getIsServer() then -- server
        if streamWriteBool(streamId, bitAND(dirtyMask, self.wheelShaderDirtyFlag) ~= 0) then
            for _, v in ipairs(self.shaderWheels) do
                streamWriteFloat32(streamId, v.currentRotation);
            end;
        end;
    end;
end;

function WheelShader:mouseEvent(posX, posY, isDown, isUp, button)
end;

function WheelShader:keyEvent(unicode, sym, modifier, isDown)
end;

function WheelShader:update(dt)
    --[[ 
    -- TODO --
        improve shader rotation speed, we shouldnt need to set it...
        
    -- Issues -- 
        Wheels are in ground on load, MP
        Deformation gets a bit "jumpy", MP
        
    -- Limitations
        currentRotation can get too big, we should reset once wheel has rotated 1 time.
    ]]--
    
    if (self:getIsActive() or self.forceShaderWheelUpdate) then
        for i, v in ipairs(self.shaderWheels) do
            local wheel = self.ourWheelParent.wheels[v.wheelId];
            local steeringAngle = wheel.steeringAngle;
            local xDrive = wheel.netInfo.xDrive;
            
            if self.isServer then
                -- Wheel Rotation --
                local axleSpeed = 0;
                local movingDirection = 0;
                
                if xDrive ~= v.lastXDrive then
                    axleSpeed = getWheelShapeAxleSpeed(wheel.node, wheel.wheelShape);
                    
                    if axleSpeed > 0.01 then
                        movingDirection = 1;
                    elseif axleSpeed < -0.01 then
                        movingDirection = -1;
                    end;
                end;
                
                local movingSpeed = math.abs(axleSpeed * (wheel.radius * math.pi));
                v.currentRotation = (v.currentRotation + movingSpeed * movingDirection * (v.rotSpeed / (v.wheelDiameter * self.rotSpeed)));
                v.lastXDrive = xDrive;
                
                if self.isClient then
                    setShaderParameter(v.node, "rotateWheel", 0, 0, 0, v.currentRotation, false);
                end;
                
                if v.currentRotation ~= v.sentRotation then
                    v.sentRotation = v.currentRotation;
                    
                    self:raiseDirtyFlags(self.wheelShaderDirtyFlag);
                end;
            end;
            
            -- Extra Rotation --
            if v.steeringNode ~= nil then
                setRotation(v.steeringNode, 0, steeringAngle, 0);
            end;
            
            if v.driveNode ~= nil then
                setRotation(v.driveNode, xDrive, 0, 0);
            end;
            
            if v.fenderNode ~= nil then
                local angleDif = 0;
                if steeringAngle > v.fenderRotMax then
                    angleDif = v.fenderRotMax - steeringAngle;
                elseif steeringAngle < v.fenderRotMin then
                    angleDif = v.fenderRotMin - steeringAngle;
                end;
                
                setRotation(v.fenderNode, 0, angleDif, 0);
            end;
            
            -- Deformation --
            v.suspensionCompression = 100 * (wheel.netInfo.y - wheel.netInfo.yMin) / wheel.suspTravel - 20; -- there might still be an short delay in MP as netInfo needs to be synched to client
            
            local masterAirPressure = Utils.getNoNil(self.wheelShaderMasterAirPressure, v.airPressure);
            if not v.useMaserAirPressureScale and self.wheelShaderMasterAirPressure ~= nil then
                masterAirPressure = v.airPressure;
            end;
            
            if (v.lastSuspensionCompression ~= v.suspensionCompression or v.sentAirPressure ~= masterAirPressure or self.forceShaderWheelUpdateOnLoad) then
                local deformation, updateWheelBase = (v.suspensionCompression * 0.01), false;
                
                if v.suspensionCompression < 0 then
                    deformation = v.lastDeformationWithGroundContact * (math.abs((math.max(v.suspensionCompression, -20) / -20) - 1));
                else
                    local airPressure = v.maxDeformation * masterAirPressure + v.maxDeformation * deformation * masterAirPressure;
                    deformation = deformation * (v.maxDeformation * self.maxImpactDeformation);
                    deformation = math.min(math.max(deformation, airPressure), v.maxDeformation);
                    v.lastDeformationWithGroundContact = deformation;
                end;
                
                v.deformation = deformation;
                v.lastSuspensionCompression = v.suspensionCompression;
                
                -- only update for clients
                if self.isClient then
                    setShaderParameter(v.node, "updateWheel", deformation, 0, v.maxDeformation, 0, false);
                end;
                
                if (v.deformation >= v.lastPositionUpdate + (WheelShader.UPDATE_TIME * 0.01) or v.deformation <= v.lastPositionUpdate - (WheelShader.UPDATE_TIME * 0.01)) then
                    v.lastPositionUpdate = v.deformation;
                    
                    if v.suspensionCompression >= 0 then
                        self:updateshaderWheelPosition(v, wheel);
                    end;
                end;
            end;
        end;
        
        if self.weAreHardPointObject then
            if self.controlUnitVersion <= 1.3 then
                self.firstTimeRun = true; -- version 1.31 does this already be default. 
            end;
        end;
        
        if self.showShaderWheelDebug and not Vehicle.debugRendering then 
            -- if self.isServer then
                local parent = self;
                if self.weAreHardPointObject then
                    parent = self.vehicle;
                end;
                
                local rootAttacherVehicle = parent:getRootAttacherVehicle();
                
                if rootAttacherVehicle.isEntered and (rootAttacherVehicle == parent or rootAttacherVehicle.shaderWheels == nil) then
                    if rootAttacherVehicle.selectedImplement ~= nil and rootAttacherVehicle.selectedImplement.object ~= nil then
                        if rootAttacherVehicle.selectedImplement.object.shaderWheels ~= nil then
                            -- this dont include HP objects
                            WheelShader.drawDebugRendering(rootAttacherVehicle.selectedImplement.object);
                        end;
                    else
                        WheelShader.drawDebugRendering(self);
                    end;
                end;
            -- end;
        end;
    end;
end;

function WheelShader:updateTick(dt)
end;

function WheelShader:draw()
end;

function WheelShader:updateshaderWheelPosition(shaderWheel, wheel)
    if wheel.backup == nil then
        wheel.backup = {wheel.spring, wheel.frictionScale, wheel.maxLongStiffness, wheel.maxLatStiffness, wheel.maxLatStiffnessLoad};
    end;
    
    -- this remove part of the impact bounc
    -- local scale = -shaderWheel.deformation / shaderWheel.maxDeformation;
    -- wheel.deltaY = (1 - scale) * wheel.suspTravel;
    
    -- flaw: if wheel deform too much then we might need to force an move of the wheel?
    local scale = Utils.getNoNil(self.wheelShaderMasterAirPressure, shaderWheel.airPressure);
    if not shaderWheel.useMaserAirPressureScale and self.wheelShaderMasterAirPressure ~= nil then
        scale = shaderWheel.airPressure;
    end;
    
    scale = -scale;
    wheel.deltaY = (1 - scale) * wheel.suspTravel;
    shaderWheel.lastPositionUpdate = shaderWheel.deformation;
    scale = scale + 1;
    
    for name, v in pairs(self.curveSettings) do
        local curve = v:get(scale);
        wheel[name] = curve * wheel.backup[name];
    end;
    
    self.ourWheelParent:updateWheelBase(wheel);
    self.ourWheelParent:updateWheelTireFriction(wheel);
end;

function WheelShader:loadDataFromXML(xmlFile)
    self.showShaderWheelDebug = Utils.getNoNil(getXMLBool(xmlFile, "vehicle.shaderWheels#debug"), false);
    
    if self.showShaderWheelDebug then
        self.isSelectable = true;
    end;
    
    self.maxImpactDeformation = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.shaderWheels#maxImpactDeformationPercentage"), 0.7); -- max impact deformation
    self.rotSpeed = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.shaderWheels#rotSpeed"), 1);
    
    if self.openPressureValve == nil then -- TirePressure script is not present!
        self.wheelShaderMasterAirPressure = getXMLFloat(xmlFile, "vehicle.shaderWheels#airPressure"); -- affect all wheels if it got an value
        
        if self.wheelShaderMasterAirPressure ~= nil then
            self.wheelShaderMasterAirPressure = math.abs(self.wheelShaderMasterAirPressure - 1);
        end;
    end;
end;

function WheelShader:loadWheelDataFromXML(xmlFile, key, entry)
    entry.affectedRadius = Utils.getNoNil(getXMLFloat(xmlFile, key .. "#affectedDiameter"), entry.wheelDiameter * 0.9) / 2;
    entry.innerRadius = Utils.getNoNil(getXMLFloat(xmlFile, key .. "#innerDiameter"), entry.wheelDiameter * 0.4) / 2;
    entry.deformationY = Utils.getNoNil(getXMLFloat(xmlFile, key .. "#deformationY"), 1.85);
    entry.centerFactor = Utils.getNoNil(getXMLFloat(xmlFile, key .. "#centerFactor"), 0.5);
    entry.maxDeformation = Utils.getNoNil(getXMLFloat(xmlFile, key .. "#maxDeformation"), 0.44);
    entry.extraPressureFactors = Utils.getNoNil(Utils.getVectorNFromString(getXMLString(xmlFile, key .. "#extraPressureFactors"), 3), {0, 0, 4});
    
    entry.airPressure = math.abs(Utils.getNoNil(getXMLFloat(xmlFile, key .. "#airPressure"), 1) - 1);
    entry.useMaserAirPressureScale = Utils.getNoNil(getXMLBool(xmlFile, key .. "#useMaserAirPressureScale"), true);
    
    entry.rotSpeed = Utils.getNoNil(getXMLFloat(xmlFile, key .. "#rotSpeed"), 1);
    
    local wheel = self.ourWheelParent.wheels[entry.wheelId];
    entry.fenderRotMax = Utils.getNoNilRad(getXMLFloat(xmlFile, key .. "#fenderRotMax"), wheel.rotMax);
    entry.fenderRotMin = Utils.getNoNilRad(getXMLFloat(xmlFile, key .. "#fenderRotMin"), wheel.rotMin);
end;

function WheelShader.developmentReloadFromXML(self, xmlFile)
    self:loadDataFromXML(xmlFile);
    
    for i = 1, table.getn(self.shaderWheels) do
        local wheel = self.shaderWheels[i];
        local key = string.format("vehicle.shaderWheels.wheel(%d)", wheel.xmlIndex);

        self:loadWheelDataFromXML(xmlFile, key, wheel);
        
        self:updateshaderWheelPosition(wheel, self.ourWheelParent.wheels[wheel.wheelId]);
    end;
end;

function WheelShader.loadCurve(self, xmlFile, curveSetting, name)
    local curve = AnimCurve:new(linearInterpolator1);
    local i = 0;
    while true do
        local key = string.format("vehicle.shaderWheels.%s(%d)", name, i);
        if not hasXMLProperty(xmlFile, key) then break;    end;
        
        local value = getXMLFloat(xmlFile, key .. "#value");
        local scale = getXMLFloat(xmlFile, key .. "#scale");
        
        if value ~= nil and scale ~= nil then
            curve:addKeyframe({v = value, time = scale});
        end;
        
        i = i + 1;
    end;
    
    for i, wheel in ipairs(self.ourWheelParent.wheels) do
        if wheel.backup == nil then
            wheel.backup = {};
        end;
        
        wheel.backup[name] = wheel[name];
    end;
        
    if i > 0 then
        curveSetting[name] = curve;
    end;
end;

function WheelShader.drawDebugRendering(self)
    local wheelsStrs = {"\n", "deltaY\n", "deformation\n", "airPressure\n", "nextUpdate\n"};
    for name, v in pairs(self.curveSettings) do
        table.insert(wheelsStrs, name .. "\n");
    end;
    table.insert(wheelsStrs, "\n");
    
    for i, v in ipairs(self.shaderWheels) do
        local wheel = self.ourWheelParent.wheels[v.wheelId];
        local air = Utils.getNoNil(self.wheelShaderMasterAirPressure, v.airPressure);
        local posUpdate = v.lastPositionUpdate * 100;
        
        wheelsStrs[1] = wheelsStrs[1] .. string.format("%d:\n", i);
        wheelsStrs[2] = wheelsStrs[2] .. string.format("%1.2f\n", wheel.deltaY);
        wheelsStrs[3] = wheelsStrs[3] .. string.format("%1.0f%%\n", (v.deformation / v.maxDeformation) * 100);
        wheelsStrs[4] = wheelsStrs[4] .. string.format("%1.0f%%\n", (1 - air) * 100);
        wheelsStrs[5] = wheelsStrs[5] .. string.format("%1.0f%% or %1.0f%% (%1.0f%%)\n", posUpdate + WheelShader.UPDATE_TIME, posUpdate - WheelShader.UPDATE_TIME, posUpdate);
        local i = 6;
        for name, _ in pairs(self.curveSettings) do
            wheelsStrs[i] = wheelsStrs[i] .. string.format("%1.0f%%\n", (wheel[name] / wheel.backup[name]) * 100);
            i = i + 1;
        end;
    end;
    
    Utils.renderMultiColumnText(0.18, 0.22, getCorrectTextSize(0.013), wheelsStrs, 0.008, {RenderText.ALIGN_RIGHT, RenderText.ALIGN_LEFT});
end;