--
-- Fliegl Overload Station
--
--
-- @author  Manuel Leithner
-- @date  26/09/13
--
-- Copyright (C) GIANTS Software GmbH, Confidential, All Rights Reserved.
--
-- Edit: Marhu und Ifko[nator] @ 28.12.2015
-- Fixes for the Fliegl Overloadstation

Overloading2 = {};

source("dataS/scripts/vehicles/specializations/OverloadingSetPipeStateEvent.lua");
source("dataS/scripts/vehicles/specializations/OverloadingToggleUnloadEvent.lua");

function Overloading2.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Fillable, specializations);
end;
function Overloading2:load(savegame)
    self.getIsFoldAllowed = Utils.overwrittenFunction(self.getIsFoldAllowed, Overloading2.getIsFoldAllowed);
    self.setPipeState = SpecializationUtil.callSpecializationsFunction("setPipeState");
    self.getOverloading2TrailerInRangePipeState = Overloading2.getOverloading2TrailerInRangePipeState;
    self.getIsPipeStateChangeAllowed = Overloading2.getIsPipeStateChangeAllowed;
    self.getIsPipeUnloadingAllowed = Overloading2.getIsPipeUnloadingAllowed;
    self.findAutoAimTrailerToUnload = Overloading2.findAutoAimTrailerToUnload;
    self.findTrailerToUnload = Overloading2.findTrailerToUnload;
    self.findTrailerRaycastCallback = Overloading2.findTrailerRaycastCallback;
    self.onOverloading2TrailerTrigger = Overloading2.onOverloading2TrailerTrigger;
    self.setOverloadingActive = Overloading2.setOverloadingActive;
    self.activatePipeUnloadingEffects = Overloading2.activatePipeUnloadingEffects;
    if hasXMLProperty(xmlFile, string.format("vehicle.combineTrailerTriggers.combineTrailerTrigger(%d)", 0)) then
        print("Warning: combineTrailerTriggers are no longer supported. Use vehicle.trailerTriggers.trailerTrigger instead");
    end
    self.trailerTriggers = {};
    local i = 0;
    while true do
        local key = string.format("vehicle.trailerTriggers.trailerTrigger(%d)", i);
        if not hasXMLProperty(xmlFile, key) then
            key = string.format("vehicle.combineTrailerTriggers.combineTrailerTrigger(%d)", i);
            if not hasXMLProperty(xmlFile, key) then
                break;
            end;
        end;
        local node = Utils.indexToObject(self.components, getXMLString(xmlFile, key.."#index"));
        if node ~= nil then
            local pipeState = Utils.getNoNil(getXMLInt(xmlFile, key.."#pipeState"), 2);
            self.trailerTriggers[node] = {node=node, pipeState=pipeState};
        end;
        i = i + 1;
    end;
    for _, trigger in pairs(self.trailerTriggers) do
        addTrigger(trigger.node, "onOverloading2TrailerTrigger", self);
    end;
    self.Overloading2TrailersInRange = {};
    self.Overloading2TriggersTrailers = {};
    self.foldMaxPipeState = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.pipe#foldMaxState"), 999);
    self.foldMinPipeState = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.pipe#foldMinState"), 0);
    self.pipeNodes = {};
    self.numPipeStates = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.pipe#numStates"), 0);
    self.currentPipeState = 1;
    self.targetPipeState = 1;
    self.pipeStateIsUnloading = {};
    self.pipeStateIsAutoAiming = {};
    local unloadingPipeStates = Utils.getVectorNFromString(getXMLString(xmlFile, "vehicle.pipe#unloadingStates"));
    if unloadingPipeStates ~= nil then
        for i=1, table.getn(unloadingPipeStates) do
            if unloadingPipeStates[i] ~= nil then
                self.pipeStateIsUnloading[unloadingPipeStates[i] ] = true;
            end;
        end;
    end;
    local autoAimPipeStates = Utils.getVectorNFromString(getXMLString(xmlFile, "vehicle.pipe#autoAimStates"));
    if autoAimPipeStates ~= nil then
        for i=1, table.getn(autoAimPipeStates) do
            if autoAimPipeStates[i] ~= nil then
                self.pipeStateIsAutoAiming[autoAimPipeStates[i] ] = true;
            end;
        end;
    end;
    local i = 0;
    while true do
        local key = string.format("vehicle.pipe.node(%d)", i);
        if not hasXMLProperty(xmlFile, key) then
            break;
        end;
        local node = Utils.indexToObject(self.components, getXMLString(xmlFile, key.."#index"));
        if node ~= nil then
            local entry = {};
            entry.node = node;
            entry.autoAimXRotation = Utils.getNoNil(getXMLBool(xmlFile, key.."#autoAimXRotation"), false);
            entry.autoAimYRotation = Utils.getNoNil(getXMLBool(xmlFile, key.."#autoAimYRotation"), false);
            entry.autoAimInvertZ = Utils.getNoNil(getXMLBool(xmlFile, key.."#autoAimInvertZ"), false);
            entry.states = {};
            for state=1,self.numPipeStates do
                local stateKey = key..string.format(".state%d", state);
                entry.states[state] = {};
                local x,y,z = Utils.getVectorFromString(getXMLString(xmlFile, stateKey.."#translation"));
                if x == nil or y == nil or z == nil then
                    x,y,z = getTranslation(node);
                else
                    if state == 1 then
                        setTranslation(node, x,y,z);
                    end;
                end;
                entry.states[state].translation = {x,y,z};
                local x,y,z = Utils.getVectorFromString(getXMLString(xmlFile, stateKey.."#rotation"));
                if x == nil or y == nil or z == nil then
                    x,y,z = getRotation(node);
                else
                    x,y,z = math.rad(x),math.rad(y),math.rad(z);
                    if state == 1 then
                        setRotation(node, x,y,z);
                    end;
                end;
                entry.states[state].rotation = {x,y,z};
            end;
            local x,y,z = Utils.getVectorFromString(getXMLString(xmlFile, key.."#translationSpeeds"));
            if x ~= nil and y ~= nil and z ~= nil then
                x,y,z = x*0.001,y*0.001,z*0.001;
                if x ~= 0 or y ~= 0 or z ~= 0 then
                    entry.translationSpeeds = {x,y,z};
                end;
            end;
            local x,y,z = Utils.getVectorFromString(getXMLString(xmlFile, key.."#rotationSpeeds"));
            if x ~= nil and y ~= nil and z ~= nil then
                x,y,z = math.rad(x)*0.001,math.rad(y)*0.001,math.rad(z)*0.001;
                if x ~= 0 or y ~= 0 or z ~= 0 then
                    entry.rotationSpeeds = {x,y,z};
                end;
            end;
            local x,y,z = getTranslation(node);
            entry.curTranslation = {x,y,z};
            local x,y,z = getRotation(node);
            entry.curRotation = {x,y,z};
            table.insert(self.pipeNodes, entry);
        end;
        i = i + 1;
    end;
    self.pipeAnimation = getXMLString(xmlFile, "vehicle.pipe#animName");
    self.pipeAnimSpeedScale = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.pipe#animSpeedScale"), 1);
    self.pipeRaycastNode = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.pipe#raycastNodeIndex"));
    self.pipeRaycastDistance = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.pipe#raycastDistance"), 7);
    self.pipeUnloadingDistance = 0;
    self.pipeParticleSystemExtraDistance = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.pipeParticleSystems#extraDistance"), 1.5);
    self.hasPipe = table.getn(self.pipeNodes) ~= 0 or self.pipeAnimation ~= nil;
    self.doAutoAiming = Utils.getNoNil(getXMLBool(xmlFile, "vehicle.pipe#doAutoAiming"), self.pipeRaycastNode == nil);
    if self.isClient then
        local defaultPipePSNode = self.components[1].node;
        if table.getn(self.pipeNodes) > 0 then
            defaultPipePSNode = self.pipeNodes[1].node;
        end
        self.pipeParticleSystems = {};
        -- load the pipe particle system for each fruit type
        local i = 0;
        while true do
            local key = string.format("vehicle.pipeParticleSystems.pipeParticleSystem(%d)", i);
            local t = getXMLString(xmlFile, key .. "#type");
            if t == nil then
                break;
            end;
            local desc = FruitUtil.fruitTypes[t];
            if desc ~= nil then
                local currentPS = Utils.getNoNil(self.pipeParticleSystems[desc.index], {});
                local particleNode = Utils.loadParticleSystem(xmlFile, currentPS, key, self.components, false, nil, self.baseDirectory, defaultPipePSNode);
                for _, v in ipairs(currentPS) do
                    local normalSpeed,tangentSpeed = getParticleSystemAverageSpeed(v.geometry);
                    v.speed = math.sqrt(normalSpeed*normalSpeed + tangentSpeed*tangentSpeed);
                    v.originalLifespan = getParticleSystemLifespan(v.geometry);
                end
                self.pipeParticleSystems[desc.index] = currentPS;
                if self.defaultPipeParticleSystem == nil then
                    self.defaultPipeParticleSystem = currentPS;
                end;
                if self.pipeRaycastNode == nil then
                    self.pipeRaycastNode = particleNode;
                end;
            end;
            i = i + 1;
        end;
        self.pipeEffect = EffectManager:loadEffect(xmlFile, "vehicle.pipeEffect", self.components, self);
        self.Overloading2TurnedOnRotationNodes = Utils.loadRotationNodes(xmlFile, {}, "vehicle.turnedOnRotationNodes.turnedOnRotationNode", "overloading", self.components);
        self.pipeScrollers = Utils.loadScrollers(self.components, xmlFile, "vehicle.pipeScrollers.pipeScroller", {}, false);
        self.samplePipe = Utils.loadSample(xmlFile, {}, "vehicle.pipeSound", nil, self.baseDirectory);
        local linkNode = Utils.indexToObject(self.components, Utils.getNoNil(getXMLString(xmlFile, "vehicle.fillSound#linkNode"), "0>"));
        self.sampleFill = Utils.loadSample(xmlFile, {}, "vehicle.fillSound", nil, self.baseDirectory, linkNode);
        Utils.stop3DSample(self.sampleFill);
        self.sampleFillEnabled = false;
        self.sampleFillStopTime = -1;
        self.lastFillLevel = -1;
        self.sampleFillUseOnDischarge = Utils.getNoNil(getXMLBool(xmlFile, "vehicle.fillSound#useOnDischarge"), false);
        self.sampleFillUseOnCharge = Utils.getNoNil(getXMLBool(xmlFile, "vehicle.fillSound#useOnCharge"), false);
    end;
    if self.pipeRaycastNode == nil then
        self.pipeRaycastNode = self.components[1].node;
    end;
    self.pipeUnloadEffectStopTime = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.pipe#unloadingEffectStopTime"), 0) * 1000;
    self.currentUnloadEffectStopTime = 0;
    self.Overloading2Capacity = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.OverloadingCapacity"), 10);
    self.pipeIsUnloading = false;
    self.pipeParticleDeactivateTime = -1;
    self.doAutoUnloadIfHired = Utils.getNoNil(getXMLBool(xmlFile, "vehicle.pipe#doAutoUnloadIfHired"), false);
    self.allowFoldWhileUnloading = Utils.getNoNil(getXMLBool(xmlFile, "vehicle.threshingFold#allowFoldWhileUnloading"), false);
    self.pipeFoldMinLimit = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.pipe#foldMinLimit"), 0);
    self.pipeFoldMaxLimit = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.pipe#foldMaxLimit"), 1);
    self.Overloading2DirtyFlag = self:getNextDirtyFlag();
    self.isOverloading2Active = false;
    self.isSelectable = true;
    self:addConflictCheckedInput(InputBinding.EMPTY_GRAIN);
    
    self.fillVolumeUnloadPlanes = {};
    
    local i = 0;
    while true do
        local key = string.format("vehicle.fillVolumes.fillVolume(%d)", i);
        if not hasXMLProperty(xmlFile, key) then
            break;
        end;
        
        local isUnloadPlane = Utils.getNoNil(getXMLBool(xmlFile, key .. "#isUnloadPlane"), false);
        
        if isUnloadPlane then
            local node = Utils.indexToObject(self.components, getXMLString(xmlFile, key .. "#index"))
            
            if node ~= nil then
                local entry = {};
                entry.node = node;
                
                setVisibility(entry.node, false);
            
                table.insert(self.fillVolumeUnloadPlanes, entry);
            end;
        end;
        
        i = i + 1;
    end;
end;
function Overloading2:delete()
    for _, trailerTrigger in pairs(self.trailerTriggers) do
        removeTrigger(trailerTrigger.node);
    end;
    if self.isClient then
        EffectManager:deleteEffect(self.pipeEffect);
        for _,v in pairs(self.pipeParticleSystems) do
            Utils.deleteParticleSystem(v);
        end;
        Utils.deleteSample(self.samplePipe);
    end;
    if self.isClient then
        Utils.deleteSample(self.sampleFill);
    end;
end;
function Overloading2:loadFromAttributesAndNodes(xmlFile, key, resetVehicles)
    if not resetVehicles then
        local pipeState = Utils.getNoNil(getXMLInt(xmlFile, key.."#pipeState"), 1);
        self:setPipeState(pipeState, true);
    end
    return BaseMission.VEHICLE_LOAD_OK;
end
function Overloading2:getSaveAttributesAndNodes(nodeIdent)
    local attributes = 'pipeState="'..self.currentPipeState..'"';
    return attributes, nil;
end
function Overloading2:readStream(streamId, connection)
    if self.hasPipe then
        local pipeState = streamReadUIntN(streamId, 3);
        local pipeUnloadingDistance = streamReadUIntN(streamId, 5);
        if pipeUnloadingDistance == 0 then
            self.pipeIsUnloading = false;
        else
            self.pipeIsUnloading = true;
            self.pipeUnloadingDistance = pipeUnloadingDistance*self.pipeRaycastDistance / 31;
        end
        self:setPipeState(pipeState, true);
    else
        self:setOverloadingActive(streamReadBool(streamId), true);
    end;
end;
function Overloading2:writeStream(streamId, connection)
    if self.hasPipe then
        streamWriteUIntN(streamId, self.targetPipeState, 3);
        if self.pipeIsUnloading then
            streamWriteUIntN(streamId, Utils.clamp(math.floor(self.pipeUnloadingDistance/self.pipeRaycastDistance*31), 1, 31), 5);
        else
            streamWriteUIntN(streamId, 0, 5);
        end;
    else
        streamWriteBool(streamId, self.isOverloading2Active);
    end;
end;
function Overloading2:readUpdateStream(streamId, timestamp, connection)
    if connection:getIsServer() then
        --if self.hasPipe then
            if streamReadBool(streamId) then
                local pipeUnloadingDistance = streamReadUIntN(streamId, 5);
                if pipeUnloadingDistance == 0 then
                    self.pipeIsUnloading = false;
                else
                    self.pipeIsUnloading = true;
                    self.pipeUnloadingDistance = pipeUnloadingDistance*self.pipeRaycastDistance / 31;
                end
            end;
        --end;
    end;
end;
function Overloading2:writeUpdateStream(streamId, connection, dirtyMask)
    if not connection:getIsServer() then
        --if self.hasPipe then
            if streamWriteBool(streamId, bitAND(dirtyMask, self.Overloading2DirtyFlag) ~= 0) then
                if self.pipeIsUnloading then
                    streamWriteUIntN(streamId, Utils.clamp(math.floor(self.pipeUnloadingDistance/self.pipeRaycastDistance*31), 1, 31), 5);
                else
                    streamWriteUIntN(streamId, 0, 5);
                end
            end;
        --end;
    end;
end;
function Overloading2:mouseEvent(posX, posY, isDown, isUp, button)
end;
function Overloading2:keyEvent(unicode, sym, modifier, isDown)
end;
function Overloading2:update(dt)
    if self:getIsActive() then
        if self.isClient and self:getIsActiveForInput(true) and not self:hasInputConflictWithSelection() then
            if InputBinding.hasEvent(InputBinding.EMPTY_GRAIN) then
                if self.hasPipe then
                    local nextState = self.targetPipeState+1;
                    if nextState > self.numPipeStates then
                        nextState = 1;
                    end;
                    if self:getIsPipeStateChangeAllowed(nextState) then
                        self:setPipeState(nextState);
                    elseif nextState ~= 1 and self:getIsPipeStateChangeAllowed(1) then
                        -- also try to close the pipe if other states are not allowed
                        self:setPipeState(1);
                    end
                else
                    self:setOverloadingActive(not self.isOverloading2Active);
                end;
            end;
        end;
        if not self.hasPipe and self.doAutoUnloadIfHired then
            local rootAttacher = self:getRootAttacherVehicle();
            if self:findTrailerToUnload(self.currentFillType) ~= nil and self.fillLevel > 0 and rootAttacher.isHired then
                self:setOverloadingActive(true);
            end;
        end;
        if self.hasPipe then
            local doAutoAiming = self.pipeStateIsAutoAiming[self.currentPipeState];
            local targetTrailer = nil;
            if doAutoAiming then
                targetTrailer = self:findAutoAimTrailerToUnload(self.lastValidFillType);
                if targetTrailer == nil then
                    doAutoAiming = false;
                end;
            end;
            if (self.currentPipeState ~= self.targetPipeState or doAutoAiming) and self.targetPipeState <= self.numPipeStates then
                local autoAimX, autoAimY, autoAimZ;
                if doAutoAiming then
                    autoAimX, autoAimY, autoAimZ = getWorldTranslation(targetTrailer.fillAutoAimTargetNode);
                end;
                local moved = false;
                for i=1, table.getn(self.pipeNodes) do
                    local nodeMoved = false;
                    local pipeNode = self.pipeNodes[i];
                    local state = pipeNode.states[self.targetPipeState];
                    if pipeNode.translationSpeeds ~= nil then
                        for i=1, 3 do
                            if math.abs(pipeNode.curTranslation[i] - state.translation[i]) > 0.000001 then
                                nodeMoved = true;
                                if pipeNode.curTranslation[i] < state.translation[i] then
                                    pipeNode.curTranslation[i] = math.min(pipeNode.curTranslation[i] + dt*pipeNode.translationSpeeds[i], state.translation[i]);
                                else
                                    pipeNode.curTranslation[i] = math.max(pipeNode.curTranslation[i] - dt*pipeNode.translationSpeeds[i], state.translation[i]);
                                end;
                            end;
                        end;
                        setTranslation(pipeNode.node, pipeNode.curTranslation[1],pipeNode.curTranslation[2],pipeNode.curTranslation[3])
                    end;
                    if pipeNode.rotationSpeeds ~= nil then
                        for i=1, 3 do
                            local targetRotation = state.rotation[i];
                            if doAutoAiming then
                                if pipeNode.autoAimXRotation and i == 1 then
                                    local x,y,z = getWorldTranslation(pipeNode.node);
                                    local x,y,z = worldDirectionToLocal(getParent(pipeNode.node), autoAimX-x, autoAimY-y, autoAimZ-z);
                                    targetRotation = -math.atan2(y,z);
                                    if pipeNode.autoAimInvertZ then
                                        targetRotation = targetRotation+math.pi;
                                    end;
                                    targetRotation = Utils.normalizeRotationForShortestPath(targetRotation, pipeNode.curRotation[i]);
                                elseif pipeNode.autoAimYRotation and i == 2 then
                                    local x,y,z = getWorldTranslation(pipeNode.node);
                                    local x,y,z = worldDirectionToLocal(getParent(pipeNode.node), autoAimX-x, autoAimY-y, autoAimZ-z);
                                    targetRotation = math.atan2(x,z);
                                    if pipeNode.autoAimInvertZ then
                                        targetRotation = targetRotation+math.pi;
                                    end;
                                    targetRotation = Utils.normalizeRotationForShortestPath(targetRotation, pipeNode.curRotation[i]);
                                end;
                            end;
                            if math.abs(pipeNode.curRotation[i] - targetRotation) > 0.000001 then
                                nodeMoved = true;
                                if pipeNode.curRotation[i] < targetRotation then
                                    pipeNode.curRotation[i] = math.min(pipeNode.curRotation[i] + dt*pipeNode.rotationSpeeds[i], targetRotation);
                                else
                                    pipeNode.curRotation[i] = math.max(pipeNode.curRotation[i] - dt*pipeNode.rotationSpeeds[i], targetRotation);
                                end;
                            end;
                        end;
                        setRotation(pipeNode.node, pipeNode.curRotation[1],pipeNode.curRotation[2],pipeNode.curRotation[3])
                    end;
                    moved = moved or nodeMoved;
                    if nodeMoved and self.setMovingToolDirty ~= nil then
                        self:setMovingToolDirty(pipeNode.node);
                    end;
                end;
                if table.getn(self.pipeNodes) == 0 and self.pipeAnimation ~= nil then
                    if self:getIsAnimationPlaying(self.pipeAnimation) then
                        moved = true;
                    end;
                end;
                if not moved then
                    self.currentPipeState = self.targetPipeState;
                end;
            end;
            if self.isClient then
                if self:getIsActiveForSound() and self.currentPipeState ~= self.targetPipeState then
                    Utils.playSample(self.samplePipe, 0, 0, nil);
                else
                    Utils.stopSample(self.samplePipe);
                end;
            end;
        end;
    end;
    if self.isClient then
        Utils.updateRotationNodes(self, self.Overloading2TurnedOnRotationNodes, dt, self.pipeIsUnloading);
        if self.pipeIsUnloading then
            local currentPipeScrollers = self.pipeScrollers; --[self.lastValidFillType];
            if currentPipeScrollers ~= self.currentPipeScrollers and self.currentPipeScrollers ~= nil then
                self.lastPipeScrollers = self.currentPipeScrollers;
            end
            self.currentPipeScrollers = currentPipeScrollers;
            if self.currentPipeScrollers ~= nil then
                Utils.updateScrollers(self.currentPipeScrollers, dt, true, true);
            end
        else
            if self.currentPipeScrollers ~= nil then
                if self.currentPipeScrollers.isRunning == true then
                    Utils.updateScrollers(self.currentPipeScrollers, dt, false, true);
                else
                    Utils.updateScrollers(self.currentPipeScrollers, dt, false, false);
                    self.currentPipeScrollers = nil;
                end;
            end
        end;
        if self.lastPipeScrollers ~= nil then
            Utils.updateScrollers(self.lastPipeScrollers, dt, false, false);
            self.lastPipeScrollers = nil;
        end;
    end;
end;
function Overloading2:updateTick(dt)
    if self.isServer then
        if self.pipeIsUnloading then
            self.currentUnloadEffectStopTime = self.currentUnloadEffectStopTime - dt;
            if self.currentUnloadEffectStopTime <= 0 then
                self.pipeIsUnloading = false;
            end;
        end;
        --self.pipeIsUnloading = false;
        if (self.hasPipe and (self.pipeStateIsUnloading[self.currentPipeState] or self.numPipeStates == 0) and self:getIsPipeUnloadingAllowed()) or (not self.hasPipe and self.isOverloading2Active) then
            if self.fillLevel > 0 then
                -- test if we should drain the grain tank
                local trailer, trailerDistance;
                local doAutoAiming = false;
                if self.hasPipe then
                    doAutoAiming = self.pipeStateIsAutoAiming[self.currentPipeState];
                else
                    doAutoAiming = self.doAutoAiming;
                end;
                if doAutoAiming then
                    trailer, _ = self:findAutoAimTrailerToUnload(self.currentFillType);
                    trailerDistance = 1;
                else
                    trailer, trailerDistance = self:findTrailerToUnload(self.currentFillType);
                end;
                if trailer == nil then
                    if self:getCapacity() == 0 then
                        self:activatePipeUnloadingEffects(); -- show particle if capacity is 0
                    end;
                    --self.pipeIsUnloading = self:getCapacity() == 0;
                    if not self.hasPipe then
                        self:setOverloadingActive(false);
                    end;
                else
                    if self.isTurnedOn then --## check if the overloadstation is turned on
                        trailer:resetFillLevelIfNeeded(self.currentFillType);
                        local deltaLevel = self.fillLevel;
                        if self:getCapacity() > 0 then
                            deltaLevel = math.min(self.fillLevel, self.Overloading2Capacity*dt/1000.0);
                        end;
                        deltaLevel = math.min(deltaLevel, trailer:getCapacity() - trailer.fillLevel);
                        if deltaLevel > 0 then
                            self:activatePipeUnloadingEffects();
                            --self.pipeIsUnloading = true;
                            self.pipeUnloadingDistance = trailerDistance;
                            local x,y,z = getWorldTranslation(self.fillVolumeDischargeInfo.node);
                            local d1x,d1y,d1z = localDirectionToWorld(self.fillVolumeDischargeInfo.node, self.fillVolumeDischargeInfo.width,0,0);
                            local d2x,d2y,d2z = localDirectionToWorld(self.fillVolumeDischargeInfo.node, 0,0,self.fillVolumeDischargeInfo.length);
                            local fillSourceStruct = {x=x,y=y,z=z, d1x=d1x,d1y=d1y,d1z=d1z, d2x=d2x,d2y=d2y,d2z=d2z};
                            trailer:setFillLevel(trailer.fillLevel+deltaLevel, self.currentFillType, false, fillSourceStruct);
                            local x,y,z = getWorldTranslation(self.fillVolumeUnloadInfo.node);
                            local d1x,d1y,d1z = localDirectionToWorld(self.fillVolumeUnloadInfo.node, self.fillVolumeUnloadInfo.width,0,0);
                            local d2x,d2y,d2z = localDirectionToWorld(self.fillVolumeUnloadInfo.node, 0,0,self.fillVolumeUnloadInfo.length);
                            local fillSourceStruct = {x=x,y=y,z=z, d1x=d1x,d1y=d1y,d1z=d1z, d2x=d2x,d2y=d2y,d2z=d2z};
                            self:setFillLevel(self.fillLevel-deltaLevel, self.currentFillType, false, fillSourceStruct);
                        end;
                    else
                        if self:getIsActiveForInput() then -- only show warning if overload station is selected
                            g_currentMission:showBlinkingWarning(g_i18n:getText("TURN_FIRST_THE_OVERLOADSTATION_ON"), 350);
                        end;
                        
                        self:setOverloadingActive(false);
                    end;
                end;
            else
                self:setOverloadingActive(false);
            end;
        end;
        if self.pipeIsUnloading ~= self.sentPipeIsUnloading or self.pipeUnloadingDistance ~= self.sentPipeUnloadingDistance then
            self:raiseDirtyFlags(self.Overloading2DirtyFlag);
            self.sentPipeIsUnloading = self.pipeIsUnloading;
            self.sentPipeUnloadingDistance = self.pipeUnloadingDistance;
        end;
    end;
    if self.isClient then
        if self.pipeIsUnloading then
            self.pipeParticleDeactivateTime = g_currentMission.time + 200;
            local currentPipeParticleSystem = self.pipeParticleSystems[self.lastValidFillType];
            if currentPipeParticleSystem == nil then
                currentPipeParticleSystem = self.defaultPipeParticleSystem;
            end;
            if currentPipeParticleSystem ~= self.currentPipeParticleSystem then
                Utils.setEmittingState(self.currentPipeParticleSystem, false);
                self.currentPipeParticleSystem = currentPipeParticleSystem;
                Utils.setEmittingState(self.currentPipeParticleSystem, true);
            end
            if self.currentPipeParticleSystem ~= nil then
                for _, v in ipairs(self.currentPipeParticleSystem) do
                    if not v.forceFullLifespan then
                        local lifespan = math.min(v.originalLifespan, (self.pipeUnloadingDistance+self.pipeParticleSystemExtraDistance)/v.speed);
                        setParticleSystemLifespan(v.geometry, lifespan, true);
                    end;
                end
            end
            if self.pipeEffect ~= nil then
                self.pipeEffect:setFillType(self.lastValidFillType);
                EffectManager:startEffect(self.pipeEffect);
            end;
            if self.fillVolumeUnloadPlanes ~= nil then
                for _, plane in pairs(self.fillVolumeUnloadPlanes) do
                    setVisibility(plane.node, true);
                end;
                
                for i = 1, table.getn(self.fillVolumes) do
                    for j = 1,3 do
                        self.fillVolumes[i].uvPosition[j] = self.fillVolumes[i].uvPosition[j] - (self.fillVolumes[i].scrollSpeedDischarge[j]*dt);
                    end;
                    
                    setShaderParameter(self.fillVolumes[i].volume, "uvOffset", self.fillVolumes[i].uvPosition[1],self.fillVolumes[i].uvPosition[2],self.fillVolumes[i].uvPosition[3],1,false);
                end;
            end;
        else
            if self.pipeParticleDeactivateTime >= 0 and self.pipeParticleDeactivateTime <= g_currentMission.time then
                self.pipeParticleDeactivateTime = -1;
                if self.currentPipeParticleSystem ~= nil then
                    Utils.setEmittingState(self.currentPipeParticleSystem, false);
                    self.currentPipeParticleSystem = nil;
                end
                if self.pipeEffect ~= nil then
                    EffectManager:stopEffect(self.pipeEffect);
                end;
                if self.fillVolumeUnloadPlanes ~= nil then
                    for _, plane in pairs(self.fillVolumeUnloadPlanes) do
                        setVisibility(plane.node, false);
                    end;
                end;
            end
        end;
    end;
    if self.isClient then
        if self.lastFillLevel ~= self.fillLevel and self.fillLevel > 0 then
            if ((self.lastFillLevel < self.fillLevel and self.sampleFillUseOnCharge) or (self.sampleFillUseOnDischarge and self.isOverloading2Active)) and self.lastFillLevel ~= -1 then
                self.sampleFillStopTime = g_currentMission.time + 150;
            end;
            self.lastFillLevel = self.fillLevel;
        end;
        if self.sampleFillStopTime > g_currentMission.time then
            if self:getIsActiveForSound(true) then
                Utils.playSample(self.sampleFill, 0, 0, nil);
                Utils.stop3DSample(self.sampleFill);
            else
                Utils.stopSample(self.sampleFill);
                Utils.play3DSample(self.sampleFill);
            end;
        else
            Utils.stopSample(self.sampleFill);
            Utils.stop3DSample(self.sampleFill);
        end;
    end;
end;
function Overloading2:draw()
    if self.isClient and self:getIsActiveForInput(true) and not self:hasInputConflictWithSelection() then
        if self.hasPipe then
            if self.numPipeStates == 2 then
                if self.targetPipeState == 2 then
                    if self:getIsPipeStateChangeAllowed(1) then
                        g_currentMission:addHelpButtonText(g_i18n:getText("Pipe_in"), InputBinding.EMPTY_GRAIN);
                    end
                else
                    if self:getIsPipeStateChangeAllowed(2) then
                        g_currentMission:addHelpButtonText(g_i18n:getText("Dump_corn"), InputBinding.EMPTY_GRAIN);
                    end
                end
            end;
        else
            if self:findAutoAimTrailerToUnload(self.currentFillType) ~= nil then
                if self.isOverloading2Active then
                    g_currentMission:addHelpButtonText(g_i18n:getText("Stop_Overloading"), InputBinding.EMPTY_GRAIN);
                else
                    g_currentMission:addHelpButtonText(g_i18n:getText("Start_Overloading"), InputBinding.EMPTY_GRAIN);
                end;
            end;
        end;
    end;
end;
function Overloading2:activatePipeUnloadingEffects()
    self.currentUnloadEffectStopTime = self.pipeUnloadEffectStopTime;
    self.pipeIsUnloading = true;
end;
function Overloading2:onDeactivateSounds()
    if self.isClient then
        Utils.stopSample(self.samplePipe, true);
    end;
end;
function Overloading2:getIsPipeStateChangeAllowed(pipeState)
    if pipeState ~= 1 and self.foldAnimTime ~= nil and (self.foldAnimTime > self.pipeFoldMaxLimit or self.foldAnimTime < self.pipeFoldMinLimit) then
        return false;
    end
    return true;
end
function Overloading2:getIsPipeUnloadingAllowed()
    if self.foldAnimTime ~= nil and (self.foldAnimTime > self.pipeFoldMaxLimit or self.foldAnimTime < self.pipeFoldMinLimit) then
        return false;
    end
    return true;
end
function Overloading2:setOverloadingActive(isActive, noEventSend)
    if isActive ~= self.isOverloading2Active then
        OverloadingToggleUnloadEvent.sendEvent(self, isActive, noEventSend);
        self.isOverloading2Active = isActive;
    end;
end;
function Overloading2:setPipeState(pipeState, noEventSend)
    pipeState = math.min(pipeState, self.numPipeStates);
    if self.targetPipeState ~= pipeState then
        if noEventSend == nil or noEventSend == false then
            if g_server ~= nil then
                g_server:broadcastEvent(OverloadingSetPipeStateEvent:new(self, pipeState));
            else
                g_client:getServerConnection():sendEvent(OverloadingSetPipeStateEvent:new(self, pipeState), nil, nil, self);
            end;
        end;
        self.targetPipeState = pipeState;
        self.currentPipeState = 0;
        if self.pipeAnimation ~= nil then
            if pipeState == 1 then
                self:playAnimation(self.pipeAnimation, -self.pipeAnimSpeedScale, self:getAnimationTime(self.pipeAnimation), true);
            else
                self:playAnimation(self.pipeAnimation, self.pipeAnimSpeedScale, self:getAnimationTime(self.pipeAnimation), true);
            end;
        end;
    end;
end;
function Overloading2:getIsFoldAllowed(superFunc)
    if not self.allowFoldWhileUnloading and  self.pipeIsUnloading then
        return false;
    end
    if self.currentPipeState > self.foldMaxPipeState or self.currentPipeState < self.foldMinPipeState then
        return false;
    end
    if superFunc ~= nil then
        return superFunc(self);
    end
    return true;
end
function Overloading2:findAutoAimTrailerToUnload(fillType)
    local trailer = nil;
    local smallestTrailerId = nil;
    local trailerIsAttached = false;
    for trailerInRange, pipeStage in pairs(self.Overloading2TrailersInRange) do
        if trailerInRange.isDeleted then
            self.Overloading2TrailersInRange[trailerInRange] = nil;
        else
            if trailerInRange:allowFillType(fillType) and trailerInRange.getAllowFillFromAir ~= nil and trailerInRange:getAllowFillFromAir() and trailerInRange.fillLevel < trailerInRange:getCapacity() then
                local id = networkGetObjectId(trailerInRange);
                -- always take the trailer with the smallest network id which is attached if possible
                -- This is deterministic and is the same on the client
                local isAttached = trailerInRange:getIsAttachedTo(self)
                if trailer == nil or (isAttached and not trailerIsAttached) or (isAttached == trailerIsAttached and id < smallestTrailerId) then
                    trailer = trailerInRange;
                    smallestTrailerId = id;
                    trailerIsAttached = isAttached
                end;
            end;
        end
    end;
    return trailer;
end;
function Overloading2:findTrailerToUnload(fillType)
    local x,y,z = getWorldTranslation(self.pipeRaycastNode);
    local dx,dy,dz = localDirectionToWorld(self.pipeRaycastNode, 0,-1,0);
    self.trailerFound = 0;
    self.trailerFoundDistance = 0;
    raycastAll(x, y, z, dx,dy,dz, "findTrailerRaycastCallback", self.pipeRaycastDistance, self);
    local trailer = g_currentMission.nodeToVehicle[self.trailerFound];
    if trailer == nil or not trailer:allowFillType(fillType) or trailer.getAllowFillFromAir == nil or not trailer:getAllowFillFromAir() or trailer.fillLevel >= trailer:getCapacity() then
        return nil;
    end;
    return trailer, self.trailerFoundDistance;
end;
function Overloading2:findTrailerRaycastCallback(transformId, x, y, z, distance)
    local vehicle = g_currentMission.nodeToVehicle[transformId];
    if vehicle ~= nil then
        if vehicle.exactFillRootNode == transformId then
            self.trailerFound = transformId;
            self.trailerFoundDistance = distance;
            return false;
        end;
    end;
    return true;
end;
function Overloading2:getOverloading2TrailerInRangePipeState()
    local maxPipeState = 0;
    for trailer, pipeState in pairs(self.Overloading2TrailersInRange) do
        if trailer.isDeleted then
            self.Overloading2TrailersInRange[trailer] = nil;
        else
            maxPipeState = math.max(maxPipeState, pipeState);
        end
    end;
    return maxPipeState;
end
function Overloading2:onOverloading2TrailerTrigger(triggerId, otherId, onEnter, onLeave, onStay, otherShapeId)
    if onEnter or onLeave then
        local trailer = g_currentMission.nodeToVehicle[otherId];
        if trailer ~= nil and trailer.fillRootNode ~= nil and trailer ~= self then
            if onEnter then
                self.Overloading2TriggersTrailers[trailer] = Utils.getNoNil(self.Overloading2TriggersTrailers[trailer], 0) + 1;
            else
                self.Overloading2TriggersTrailers[trailer] = math.max(Utils.getNoNil(self.Overloading2TriggersTrailers[trailer], 0) - 1, 0);
            end;
            if self.Overloading2TriggersTrailers[trailer] > 0 then
                self.Overloading2TrailersInRange[trailer] = self.trailerTriggers[triggerId].pipeState;
            else
                self.Overloading2TrailersInRange[trailer] = nil;
            end;
        end;
    end;
end;